<?php
// product.php - صفحة عرض المنتج للعميل
require_once 'includes/functions.php';

// 1. استقبال رقم المنتج من الرابط
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: index.php");
    exit;
}

 $id = (int)$_GET['id'];

// 2. جلب بيانات المنتج
 $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
 $stmt->execute([$id]);
 $product = $stmt->fetch();

// لو المنتج مش موجود، ارجع للرئيسية
if (!$product) {
    header("Location: index.php");
    exit;
}

// 3. جلب صور الألبوم (Gallery)
 $stmt_g = $pdo->prepare("SELECT * FROM product_gallery WHERE product_id = ?");
 $stmt_g->execute([$id]);
 $gallery = $stmt_g->fetchAll();

// 4. جلب منتجات ذات صلة (من نفس القسم)
 $related = [];
if ($product['category_id'] > 0) {
    $stmt_r = $pdo->prepare("SELECT * FROM products WHERE category_id = ? AND id != ? LIMIT 4");
    $stmt_r->execute([$product['category_id'], $id]);
    $related = $stmt_r->fetchAll();
}

// إعدادات للعرض
 $currency = getSetting('currency');
 $whatsapp = getSetting('whatsapp');
 $current_lang = $_SESSION['lang'] ?? 'ar';
 $dir = ($current_lang == 'ar') ? 'rtl' : 'ltr';

// معالجة بيانات المخزون
 $price = (float)$product['price'];
 $old_price = isset($product['old_price']) ? (float)$product['old_price'] : 0;
 $has_discount = ($old_price > $price);

 $mainData = json_decode($product['custom_sizes'], true);
 $isNewSystem = false;
if (is_array($mainData) && isset($mainData[0]['color'])) {
    $isNewSystem = true;
}

 $colorsData = [];
 $sizesData = [];

if ($isNewSystem) {
    $tempColors = [];
    foreach ($mainData as $m) {
        if (isset($m['color'])) {
            if (!isset($tempColors[$m['color']])) $tempColors[$m['color']] = ['color' => $m['color'], 'qty' => 0];
            $tempColors[$m['color']]['qty'] += (int)($m['qty'] ?? 0);
        }
    }
    $colorsData = array_values($tempColors);
    
    $tempSizes = [];
    foreach ($mainData as $m) {
        if (isset($m['size'])) {
            if (!isset($tempSizes[$m['size']])) $tempSizes[$m['size']] = ['size' => $m['size'], 'qty' => 0];
            $tempSizes[$m['size']]['qty'] += (int)($m['qty'] ?? 0);
        }
    }
    $sizesData = array_values($tempSizes);
} else {
    $rawColors = !empty($product['custom_colors']) ? json_decode($product['custom_colors'], true) : [];
    if (is_array($rawColors)) {
        foreach ($rawColors as $c) {
            if (is_array($c) && isset($c['color'])) $colorsData[] = $c;
            elseif (is_string($c)) $colorsData[] = ['color' => $c, 'qty' => 999];
        }
    }
    $rawSizes = !empty($product['custom_sizes']) ? json_decode($product['custom_sizes'], true) : [];
    if (is_array($rawSizes)) {
        foreach ($rawSizes as $s) {
            if (is_array($s) && isset($s['qty'])) $sizesData[] = $s;
            else $sizesData[] = ['size' => trim($s), 'qty' => 999];
        }
    }
}

 $alpineData = htmlspecialchars(json_encode([
    'id' => $product['id'],
    'title' => $product['title'],
    'price' => $price,
    'image' => 'assets/uploads/' . $product['image'],
    'colors' => $colorsData,
    'sizes' => $sizesData,
]), ENT_QUOTES, 'UTF-8');

 $select_msg = ($current_lang == 'ar') ? 'الرجاء اختيار اللون والمقاس أولاً' : 'Please select color and size first';
?>

<!DOCTYPE html>
<html lang="<?= $current_lang ?>" dir="<?= $dir ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($product['title']) ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700;800&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>
    
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        sans: ['Cairo', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50: '#f0fdf4', 100: '#dcfce7', 500: '#22c55e', 600: '#16a34a', 900: '#14532d',
                        }
                    }
                }
            }
        }
    </script>

    <style>
        body { font-family: 'Cairo', sans-serif; }
        .hide-scroll::-webkit-scrollbar { display: none; }
        .hide-scroll { -ms-overflow-style: none; scrollbar-width: none; }
        .zoom-box { overflow: hidden; cursor: crosshair; position: relative; }
        .zoom-box img { transition: transform 0.3s ease-out; width: 100%; height: 100%; object-fit: contain; }
        .zoom-box:hover img { transform: scale(1.5); }
        .thumb-active { border-color: #10b981; opacity: 1; }
    </style>
</head>

<body class="bg-slate-50 text-slate-800" 
      x-data="app()"
      x-init="whatsappNumber = '<?= $whatsapp ?>'; currencySymbol = '<?= $currency ?>'; initApp();">

    <nav class="bg-white shadow-sm sticky top-0 z-40">
        <div class="container mx-auto px-4 py-3 flex justify-between items-center">
            <a href="index.php" class="text-xl font-bold text-emerald-600 flex items-center gap-2 hover:opacity-80 transition">
                <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/></svg>
                الرئيسية
            </a>
            <div class="flex items-center gap-3">
                <span class="font-bold text-gray-700">تفاصيل المنتج</span>
                <button @click="cartOpen = true" class="relative group p-2 rounded-xl hover:bg-slate-100 transition-colors">
                    <span x-show="count > 0" class="absolute -top-1 -right-1 bg-red-500 text-white text-[10px] font-bold w-5 h-5 flex items-center justify-center rounded-full border border-white" x-text="count"></span>
                    <svg class="h-6 w-6 text-slate-600 group-hover:text-emerald-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                    </svg>
                </button>
            </div>
        </div>
    </nav>

    <div class="container mx-auto px-4 py-8">
        <div class="bg-white rounded-3xl shadow-lg border border-slate-100 overflow-hidden mb-12">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-0">
                <div class="p-6 bg-slate-50 flex flex-col gap-4 relative">
                    <div class="aspect-square bg-white rounded-xl border shadow-sm flex items-center justify-center p-2 zoom-box group">
                        <img id="mainImage" src="assets/uploads/<?= htmlspecialchars($product['image']) ?>" alt="صورة المنتج">
                    </div>
                    <?php if(count($gallery) > 0): ?>
                    <div class="flex gap-3 overflow-x-auto hide-scroll pb-2">
                        <button onclick="changeImage('assets/uploads/<?= $product['image'] ?>', this)" class="thumb-btn w-20 h-20 flex-shrink-0 border-2 border-emerald-500 rounded-lg overflow-hidden bg-white p-1 cursor-pointer transition hover:opacity-100 thumb-active">
                            <img src="assets/uploads/<?= $product['image'] ?>" class="w-full h-full object-cover rounded">
                        </button>
                        <?php foreach($gallery as $gImg): ?>
                        <button onclick="changeImage('assets/uploads/<?= $gImg['image'] ?>', this)" class="thumb-btn w-20 h-20 flex-shrink-0 border-2 border-transparent rounded-lg overflow-hidden bg-white p-1 cursor-pointer opacity-70 hover:opacity-100 transition hover:border-emerald-300">
                            <img src="assets/uploads/<?= $gImg['image'] ?>" class="w-full h-full object-cover rounded">
                        </button>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>

                <div x-data="productItem(<?= $alpineData ?>)" class="p-6 md:p-12 flex flex-col">
                    <h1 class="text-3xl md:text-4xl font-extrabold text-slate-900 mb-4 leading-tight"><?= htmlspecialchars($product['title']) ?></h1>
                    <div class="flex items-center gap-4 mb-6">
                        <div class="text-4xl font-bold text-emerald-600"><?= number_format($product['price'], 2) ?> <span class="text-lg text-slate-500"><?= $currency ?></span></div>
                        <?php if($has_discount): ?>
                        <span class="text-xl text-red-400 line-through"><?= number_format($old_price, 2) ?></span>
                        <span class="bg-red-100 text-red-600 text-xs font-bold px-2 py-1 rounded">خصم</span>
                        <?php endif; ?>
                    </div>
                    <div class="text-slate-600 mb-8 leading-loose text-lg border-t border-b py-6 border-dashed border-slate-200"><?= nl2br(htmlspecialchars($product['description'])) ?></div>

                    <div class="mb-8 space-y-6">
                        <div x-show="item.colors && item.colors.length > 0">
                            <span class="text-sm font-bold text-slate-400 uppercase tracking-wider block mb-3">اختر اللون</span>
                            <div class="flex flex-wrap gap-3">
                                <template x-for="colorObj in item.colors">
                                    <button @click="if(colorObj.qty > 0) selectedColor = colorObj.color" :disabled="colorObj.qty <= 0" :class="{'ring-2 ring-offset-2 ring-emerald-500 scale-110 shadow-lg': selectedColor === colorObj.color, 'opacity-50 cursor-not-allowed': colorObj.qty <= 0}" class="w-10 h-10 rounded-full border-2 border-slate-200 transition-all duration-200 relative focus:outline-none" :style="'background-color: ' + colorObj.color">
                                        <span x-show="colorObj.qty <= 0" class="absolute inset-0 flex items-center justify-center text-red-500 font-bold text-xs line-through">x</span>
                                    </button>
                                </template>
                            </div>
                        </div>
                        <div x-show="item.sizes && item.sizes.length > 0">
                            <span class="text-sm font-bold text-slate-400 uppercase tracking-wider block mb-3">اختر المقاس</span>
                            <div class="flex flex-wrap gap-3">
                                <template x-for="s in item.sizes">
                                    <button @click="if(s.qty > 0) selectedSize = s.size" :disabled="s.qty <= 0" :class="{'bg-emerald-600 text-white border-emerald-600 shadow-md': selectedSize === s.size, 'bg-white text-slate-600 border-slate-200 hover:border-emerald-400': selectedSize !== s.size && s.qty > 0, 'bg-slate-50 text-slate-300 border-slate-100 cursor-not-allowed line-through': s.qty <= 0}" class="px-5 py-2 rounded-xl border font-bold text-sm transition-all duration-200">
                                        <span x-text="s.size"></span>
                                    </button>
                                </template>
                            </div>
                        </div>
                    </div>
                    <div class="mt-auto">
                        <button @click="addToCart()" class="w-full bg-slate-900 text-white py-4 rounded-xl font-bold text-xl hover:bg-emerald-600 transition-all duration-300 shadow-xl shadow-slate-900/20 active:scale-95 flex items-center justify-center gap-3">
                            <span>إضافة للسلة</span>
                            <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z" /></svg>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <?php if(!empty($related)): ?>
        <div class="mt-16">
            <h3 class="text-2xl font-bold text-slate-800 mb-6 flex items-center gap-2"><span class="w-2 h-8 bg-emerald-500 rounded-full"></span>منتجات مشابهة قد تعجبك</h3>
            <div class="grid grid-cols-2 md:grid-cols-4 gap-6">
                <?php foreach($related as $r): ?>
                <a href="product.php?id=<?= $r['id'] ?>" class="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden hover:shadow-lg transition group">
                    <div class="aspect-square bg-slate-50 overflow-hidden">
                        <img src="assets/uploads/<?= htmlspecialchars($r['image']) ?>" class="w-full h-full object-cover group-hover:scale-110 transition duration-500">
                    </div>
                    <div class="p-4">
                        <h4 class="font-bold text-slate-800 truncate mb-1"><?= htmlspecialchars($r['title']) ?></h4>
                        <p class="text-emerald-600 font-bold"><?= number_format($r['price'], 2) ?> <?= $currency ?></p>
                    </div>
                </a>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- سلة التسوق -->
    <div x-show="cartOpen" x-cloak class="relative z-50">
        <div x-show="cartOpen" x-transition.opacity class="fixed inset-0 bg-slate-900/40 backdrop-blur-sm" @click="cartOpen = false"></div>
        <div class="fixed inset-0 overflow-hidden pointer-events-none">
            <div class="absolute inset-0 overflow-hidden">
                <div class="pointer-events-none fixed inset-y-0 <?= ($dir == 'rtl') ? 'left-0' : 'right-0' ?> flex max-w-full pl-0 sm:pl-10">
                    <div x-show="cartOpen" x-transition:enter="transform transition ease-in-out duration-300" x-transition:enter-start="<?= ($dir == 'rtl') ? '-translate-x-full' : 'translate-x-full' ?>" x-transition:enter-end="translate-x-0" x-transition:leave="transform transition ease-in-out duration-300" x-transition:leave-start="translate-x-0" x-transition:leave-end="<?= ($dir == 'rtl') ? '-translate-x-full' : 'translate-x-full' ?>" class="pointer-events-auto w-screen max-w-md">
                        <div class="flex h-full flex-col bg-white shadow-2xl">
                            <div class="flex items-center justify-between px-6 py-5 border-b border-slate-100 bg-white z-10">
                                <h2 class="text-lg font-bold text-slate-800 flex items-center gap-3">السلة <span class="bg-emerald-50 text-emerald-700 text-xs px-2.5 py-1 rounded-full font-bold border border-emerald-100" x-text="count + ' منتج'"></span></h2>
                                <button @click="cartOpen = false" class="text-slate-400 hover:text-red-500 bg-slate-50 hover:bg-red-50 p-2 rounded-full"><svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" /></svg></button>
                            </div>
                            <div class="flex-1 overflow-y-auto p-6 bg-slate-50/50">
                                <template x-if="cart.length === 0"><div class="h-full flex flex-col items-center justify-center text-slate-400 opacity-70 gap-4"><svg class="w-12 h-12 text-slate-300" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"></path></svg><p class="font-medium">السلة فارغة</p></div></template>
                                <ul class="space-y-4">
                                    <template x-for="(item, index) in cart" :key="index">
                                        <li class="bg-white p-3 rounded-2xl border border-slate-100 shadow-sm flex gap-4">
                                            <div class="w-20 h-20 flex-shrink-0 bg-slate-100 rounded-xl overflow-hidden relative"><img :src="item.image" class="w-full h-full object-cover" onerror="this.src='https://placehold.co/100x100?text=IMG'"></div>
                                            <div class="flex-1 flex flex-col justify-between">
                                                <div>
                                                    <div class="flex justify-between items-start"><h4 class="font-bold text-slate-800 text-sm line-clamp-1" x-text="item.title"></h4><button @click="remove(index)" class="text-slate-300 hover:text-red-500"><svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg></button></div>
                                                    <div class="flex flex-wrap gap-2 mt-1.5">
                                                        <template x-if="item.variant_color"><span class="text-[10px] bg-slate-50 border border-slate-100 text-slate-500 px-1.5 py-0.5 rounded flex items-center gap-1"><span class="w-2 h-2 rounded-full inline-block ring-1 ring-slate-200" :style="'background-color:'+item.variant_color"></span><span x-text="item.variant_color"></span></span></template>
                                                        <template x-if="item.variant_size"><span class="text-[10px] bg-slate-50 border border-slate-100 text-slate-500 px-1.5 py-0.5 rounded" x-text="item.variant_size"></span></template>
                                                    </div>
                                                </div>
                                                <div class="flex justify-between items-end mt-2">
                                                    <div class="flex items-center bg-slate-50 rounded-lg border border-slate-200 h-8">
                                                        <button @click="updateQty(index, -1)" class="w-8 h-full flex items-center justify-center text-slate-500 hover:bg-slate-200 rounded-s-lg">-</button>
                                                        <span class="px-2 text-xs font-bold text-slate-800 w-6 text-center" x-text="item.qty"></span>
                                                        <button @click="updateQty(index, 1)" class="w-8 h-full flex items-center justify-center text-slate-500 hover:bg-slate-200 rounded-e-lg">+</button>
                                                    </div>
                                                    <span class="text-sm font-bold text-emerald-600" x-text="(item.price * item.qty).toFixed(2) + ' ' + currencySymbol"></span>
                                                </div>
                                            </div>
                                        </li>
                                    </template>
                                </ul>
                            </div>
                            <div class="border-t border-slate-100 bg-white p-6 shadow-[0_-10px_40px_rgba(0,0,0,0.05)] z-20">
                                <div class="flex justify-between items-center mb-6"><span class="text-slate-500 text-sm font-medium">الإجمالي</span><span class="text-2xl font-extrabold text-slate-900" x-text="total + ' ' + currencySymbol"></span></div>
                                <div class="space-y-4 mb-5">
                                    <input type="text" x-model="name" placeholder="الاسم" class="w-full bg-slate-50 px-4 py-3 border border-slate-200 rounded-xl focus:bg-white focus:border-emerald-500 outline-none transition-all duration-200 text-sm">
                                    <input type="tel" x-model="phone" placeholder="رقم الهاتف" class="w-full bg-slate-50 px-4 py-3 border border-slate-200 rounded-xl focus:bg-white focus:border-emerald-500 outline-none transition-all duration-200 text-sm">
                                    <textarea x-model="address" placeholder="العنوان" rows="2" class="w-full bg-slate-50 px-4 py-3 border border-slate-200 rounded-xl focus:bg-white focus:border-emerald-500 outline-none transition-all duration-200 text-sm resize-none"></textarea>
                                </div>
                                <button @click="smartCheckout" :disabled="cart.length === 0 || isProcessing" class="w-full bg-slate-900 text-white py-4 rounded-xl font-bold hover:bg-emerald-600 transition-all disabled:bg-slate-300 flex justify-center items-center gap-3"><span x-show="!isProcessing">إتمام الطلب</span><span x-show="isProcessing">جاري المعالجة...</span></button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('alpine:init', () => {
            Alpine.data('productItem', (data) => ({
                item: data,
                selectedColor: null,
                selectedSize: null,
                init() {
                    if(this.item.colors && this.item.colors.length === 1 && this.item.colors[0].qty > 0) this.selectedColor = this.item.colors[0].color;
                    if(this.item.sizes && this.item.sizes.length === 1 && this.item.sizes[0].qty > 0) this.selectedSize = this.item.sizes[0].size;
                },
                addToCart() {
                    if (this.item.colors && this.item.colors.length > 0 && !this.selectedColor) { alert('<?= $select_msg ?>'); return; }
                    if (this.item.sizes && this.item.sizes.length > 0 && !this.selectedSize) { alert('<?= $select_msg ?>'); return; }
                    if (this.selectedColor) { const colorObj = this.item.colors.find(c => c.color === this.selectedColor); if (colorObj && colorObj.qty <= 0) { alert('عذراً، هذا اللون غير متوفر في المخزون حالياً'); return; } }
                    if (this.selectedSize) { const sizeObj = this.item.sizes.find(s => s.size === this.selectedSize); if (sizeObj && sizeObj.qty <= 0) { alert('عذراً، هذا المقاس غير متوفر في المخزون حالياً'); return; } }
                    this.$dispatch('add-to-cart', { id: this.item.id, title: this.item.title, price: this.item.price, image: this.item.image, variant_color: this.selectedColor, variant_size: this.selectedSize });
                }
            }));
            Alpine.data('app', () => ({
                cartOpen: false,
                cart: JSON.parse(localStorage.getItem('cart')) || [],
                name: localStorage.getItem('user_name') || '',
                phone: localStorage.getItem('user_phone') || '',
                address: localStorage.getItem('user_address') || '',
                isProcessing: false,
                get count() { return this.cart.reduce((a, b) => a + b.qty, 0); },
                get total() { return this.cart.reduce((a, b) => a + (b.price * b.qty), 0).toFixed(2); },
                initApp() { window.addEventListener('add-to-cart', (e) => this.add(e.detail)); },
                add(data) {
                    let item = this.cart.find(i => i.id === data.id && i.variant_color === data.variant_color && i.variant_size === data.variant_size);
                    if (item) item.qty++;
                    else this.cart.push({ ...data, qty: 1 });
                    this.save();
                    this.cartOpen = true;
                },
                remove(index) { this.cart.splice(index, 1); this.save(); },
                updateQty(index, change) { if (this.cart[index].qty + change > 0) this.cart[index].qty += change; else this.remove(index); this.save(); },
                save() { localStorage.setItem('cart', JSON.stringify(this.cart)); },
                smartCheckout() {
                    if (!this.name || !this.address || !this.phone) { alert('يرجى إدخال الاسم ورقم الهاتف والعنوان'); return; }
                    this.isProcessing = true;
                    localStorage.setItem('user_name', this.name);
                    localStorage.setItem('user_phone', this.phone);
                    localStorage.setItem('user_address', this.address);
                    let msg = `🛒 *طلب جديد*\n\n👤 *الاسم:* ${this.name}\n📱 *رقم الهاتف:* ${this.phone}\n📍 *العنوان:* ${this.address}\n📅 *التاريخ:* ${new Date().toLocaleString('ar-EG')}\n------------------------\n📦 *تفاصيل الطلب:*`;
                    this.cart.forEach((item, index) => {
                        msg += `\n\n${index + 1}. *${item.title}*`;
                        let productUrl = window.location.origin + window.location.pathname.replace('product.php', 'index.php') + "?id=" + item.id;
                        msg += `\n🔗 الرابط: ${productUrl}`;
                        let details = [];
                        if(item.variant_size) details.push(`المقاس: ${item.variant_size}`);
                        if(item.variant_color) details.push(`اللون: ${item.variant_color}`);
                        if(details.length > 0) msg += `\n📝 التفاصيل: ${details.join(' | ')}`;
                        let itemPrice = parseFloat(item.price).toFixed(2);
                        let itemTotal = (parseFloat(item.price) * item.qty).toFixed(2);
                        msg += `\n💰 السعر: ${itemPrice} ${currencySymbol} x ${item.qty} = ${itemTotal} ${currencySymbol}`;
                    });
                    msg += `\n\n------------------------\n💰 *الإجمالي:* ${this.total} ${currencySymbol}`;
                    let formData = new FormData();
                    formData.append('action', 'submit_order');
                    formData.append('name', this.name);
                    formData.append('phone', this.phone);
                    formData.append('address', this.address);
                    formData.append('cart', JSON.stringify(this.cart));
                    formData.append('total', this.total);
                    formData.append('wa_message', msg);
                    
                    // *** التعديل الجوهري: حساب الرابط تلقائياً ***
                    // نأخذ الرابط الحالي ونستبدل اسم الملف بـ index.php
                    // هذا يضمن الوصول للسيرفر سواء كان الموقع في مجلد رئيسي أو فرعي
                    let fetchUrl = window.location.origin + window.location.pathname.replace('product.php', 'index.php');

                    fetch(fetchUrl, { method: 'POST', body: formData })
                    .then(r => r.json())
                    .then(() => {
                        window.open(`https://wa.me/${whatsappNumber}?text=${encodeURIComponent(msg)}`, '_blank');
                        this.cart = []; this.save(); this.cartOpen = false; this.isProcessing = false;
                    })
                    .catch((error) => {
                        console.log("Error:", error);
                        window.open(`https://wa.me/${whatsappNumber}?text=${encodeURIComponent(msg)}`, '_blank');
                        this.isProcessing = false;
                    });
                }
            }));
        });
        function changeImage(src, btn) {
            const mainImg = document.getElementById('mainImage');
            mainImg.parentElement.style.opacity = '0.7';
            setTimeout(() => { mainImg.src = src; mainImg.parentElement.style.opacity = '1'; }, 150);
            document.querySelectorAll('.thumb-btn').forEach(b => { b.classList.remove('border-emerald-500', 'thumb-active'); b.classList.add('border-transparent', 'opacity-70'); });
            btn.classList.remove('border-transparent', 'opacity-70');
            btn.classList.add('border-emerald-500', 'thumb-active');
        }
        const zoomBox = document.querySelector('.zoom-box');
        if(zoomBox) {
            const img = zoomBox.querySelector('img');
            zoomBox.addEventListener('mousemove', function(e) {
                const { left, top, width, height } = this.getBoundingClientRect();
                const x = (e.clientX - left) / width;
                const y = (e.clientY - top) / height;
                img.style.transformOrigin = `${x * 100}% ${y * 100}%`;
                img.style.transform = "scale(1.5)";
            });
            zoomBox.addEventListener('mouseleave', function() {
                img.style.transform = "scale(1)";
                setTimeout(() => { img.style.transformOrigin = "center center"; }, 300);
            });
        }
    </script>
</body>
</html>