<?php
/*
Plugin Name: Analytics & CRM Ultimate (Payment & Fixes)
Plugin Key: analytics_crm_ultimate
Description: نظام CRM متكامل يسجل تفاصيل الطلبات، الصور، وطريقة الدفع، مع إدارة دقيقة لحالة الطلبات.
*/

// ============================================================
// 1. التأسيس وإنشاء الجداول
// ============================================================
global $pdo;
if ($pdo) {
    
    // جدول الجلسات
    $pdo->exec("CREATE TABLE IF NOT EXISTS `acrm_sessions` (
        `session_id` varchar(64) NOT NULL,
        `ip_address` varchar(45) DEFAULT NULL,
        `last_activity` DATETIME DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`session_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // جدول العملاء
    $pdo->exec("CREATE TABLE IF NOT EXISTS `acrm_customers` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `phone` varchar(20) NOT NULL,
        `name` varchar(100) DEFAULT NULL,
        `address` text DEFAULT NULL,
        `total_orders` int(11) DEFAULT 0,
        `total_spent` decimal(10,2) DEFAULT 0.00,
        `last_order_date` DATETIME DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`), UNIQUE KEY `phone` (`phone`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
    
    // جدول الطلبات (تم إضافة payment_method)
    $pdo->exec("CREATE TABLE IF NOT EXISTS `acrm_orders` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `customer_phone` varchar(20) NOT NULL,
        `total_amount` decimal(10,2) NOT NULL,
        `payment_method` varchar(50) DEFAULT 'cod', /* عمود طريقة الدفع */
        `order_date` DATETIME DEFAULT CURRENT_TIMESTAMP,
        `status` varchar(20) DEFAULT 'pending',
        PRIMARY KEY (`id`),
        KEY `customer_phone` (`customer_phone`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // جدول تفاصيل الطلبات
    $pdo->exec("CREATE TABLE IF NOT EXISTS `acrm_order_items` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `order_id` int(11) NOT NULL,
        `product_id` int(11) NOT NULL,
        `product_name` varchar(255) DEFAULT NULL,
        `quantity` int(11) DEFAULT 1,
        `price` decimal(10,2) DEFAULT 0.00,
        `item_color` varchar(50) DEFAULT NULL,
        `item_size` varchar(50) DEFAULT NULL,
        `item_image` varchar(255) DEFAULT NULL,
        PRIMARY KEY (`id`),
        KEY `order_id` (`order_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // تحديث الجداول القديمة إن وجدت (لضمان عدم حدوث أخطاء)
    try { $pdo->exec("ALTER TABLE `acrm_order_items` ADD COLUMN `item_color` varchar(50) DEFAULT NULL;"); } catch(Exception $e) {}
    try { $pdo->exec("ALTER TABLE `acrm_order_items` ADD COLUMN `item_size` varchar(50) DEFAULT NULL;"); } catch(Exception $e) {}
    try { $pdo->exec("ALTER TABLE `acrm_order_items` ADD COLUMN `item_image` varchar(255) DEFAULT NULL;"); } catch(Exception $e) {}
    // إضافة عمود الدفع للطلبات القديمة
    try { $pdo->exec("ALTER TABLE `acrm_orders` ADD COLUMN `payment_method` varchar(50) DEFAULT 'cod';"); } catch(Exception $e) {}
}

// ============================================================
// 2. كود الاعتراض (Force View)
// ============================================================
if (isset($_GET['tool']) && $_GET['tool'] == 'analytics_crm_ultimate') {
    while (ob_get_level()) { ob_end_clean(); }
    
    if (isset($_GET['ajax_type'])) {
        acrm_handle_ajax();
        exit;
    }

    acrm_render_page_ultimate();
    die(); 
}

// ============================================================
// 3. القائمة الجانبية
// ============================================================
function acrm_sidebar_btn_ultimate() {
    $active = (isset($_GET['tool']) && $_GET['tool'] == 'analytics_crm_ultimate') ? 'bg-blue-600 text-white shadow-lg' : 'text-slate-300 hover:bg-slate-800 hover:text-white';
    echo '<a href="plugins.php?tool=analytics_crm_ultimate" class="flex items-center gap-3 px-4 py-3 rounded-xl transition '.$active.'">
            <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" /></svg>
            <span>إدارة الطلبات CRM</span>
          </a>';
}
add_action('admin_sidebar', 'acrm_sidebar_btn_ultimate');

// ============================================================
// 4. منطق حفظ البيانات
// ============================================================

// تتبع الزوار
function acrm_track_visit_ultimate() {
    global $pdo;
    if (strpos($_SERVER['REQUEST_URI'], '/admin') !== false) return;
    
    if (!isset($_COOKIE['acrm_sid'])) {
        $sid = bin2hex(random_bytes(16));
        setcookie('acrm_sid', $sid, time() + (86400 * 30), "/");
    } else { $sid = $_COOKIE['acrm_sid']; }
    
    try {
        $stmt = $pdo->prepare("INSERT INTO acrm_sessions (session_id, ip_address, last_activity) VALUES (?, ?, NOW()) ON DUPLICATE KEY UPDATE last_activity = NOW()");
        $stmt->execute([$sid, $_SERVER['REMOTE_ADDR']]);
    } catch(Exception $e) {}
}
add_action('site_head', 'acrm_track_visit_ultimate');

// حفظ الطلب الجديد (تم التعديل لحفظ طريقة الدفع)
function acrm_save_order_ultimate($order) {
    global $pdo;
    $phone = $order['phone'] ?? '';
    $name = $order['name'] ?? 'عميل';
    $total = $order['total'] ?? 0;
    $address = $order['address'] ?? '';
    // استقبال طريقة الدفع
    $payment = $order['payment_method'] ?? 'cod'; 
    
    if(empty($phone)) return;

    try {
        $stmt = $pdo->prepare("INSERT INTO acrm_customers (phone, name, address, total_orders, total_spent, last_order_date) 
            VALUES (?, ?, ?, 1, 0, NOW()) 
            ON DUPLICATE KEY UPDATE 
            name=VALUES(name), address=VALUES(address), total_orders=total_orders+1, last_order_date=NOW()");
        $stmt->execute([$phone, $name, $address]);

        // تم إضافة عمود payment_method في الإدخال
        $stmt = $pdo->prepare("INSERT INTO acrm_orders (customer_phone, total_amount, payment_method, order_date, status) VALUES (?, ?, ?, NOW(), 'pending')");
        $stmt->execute([$phone, $total, $payment]);
        $order_id = $pdo->lastInsertId();

        $items = $order['items'] ?? ($order['cart'] ?? []);

        if (!empty($items) && is_array($items)) {
            $itemStmt = $pdo->prepare("INSERT INTO acrm_order_items (order_id, product_id, product_name, quantity, price, item_color, item_size, item_image) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            
            foreach ($items as $item) {
                $p_id = $item['id'] ?? 0;
                $p_name = $item['title'] ?? ($item['name'] ?? 'منتج');
                $p_qty = $item['qty'] ?? 1;
                $p_price = $item['price'] ?? 0;
                $p_color = $item['color'] ?? null;
                $p_size = $item['size'] ?? null;
                $p_image = $item['image'] ?? null; // حفظ مسار الصورة

                $itemStmt->execute([$order_id, $p_id, $p_name, $p_qty, $p_price, $p_color, $p_size, $p_image]);
            }
        }
    } catch(Exception $e) {}
}
add_action('on_order_created', 'acrm_save_order_ultimate');

// ============================================================
// 5. معالجة AJAX (API)
// ============================================================
function acrm_handle_ajax() {
    error_reporting(E_ALL);
    ini_set('display_errors', 0); 
    header('Content-Type: application/json');
    
    global $pdo;
    $type = $_GET['ajax_type'] ?? '';
    
    try {
        if ($type == 'get_customer_orders') {
            $phone = $_GET['phone'] ?? '';
            
            $stmt = $pdo->prepare("SELECT * FROM acrm_orders WHERE customer_phone = ? ORDER BY order_date DESC");
            $stmt->execute([$phone]);
            $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach($orders as &$ord) {
                $itmStmt = $pdo->prepare("SELECT * FROM acrm_order_items WHERE order_id = ?");
                $itmStmt->execute([$ord['id']]);
                $items = $itmStmt->fetchAll(PDO::FETCH_ASSOC);

                // منطق الصور المتقدم (للتأكد من ظهورها)
                foreach($items as &$itm) {
                    if (empty($itm['item_image']) || $itm['item_image'] == 'null') {
                        if ($itm['product_id'] > 0) {
                            $prodImg = $pdo->query("SELECT image FROM products WHERE id = " . (int)$itm['product_id'])->fetchColumn();
                            if ($prodImg) {
                                $itm['item_image'] = $prodImg; 
                            }
                        }
                    }
                }
                $ord['items'] = $items;
            }
            echo json_encode($orders);
        }
        
        elseif ($type == 'update_order_status') {
            $oid = $_POST['order_id'] ?? 0;
            $status = $_POST['status'] ?? 'pending';
            $phone = $_POST['phone'] ?? '';
            
            $pdo->prepare("UPDATE acrm_orders SET status = ? WHERE id = ?")->execute([$status, $oid]);
            
            // تحديث إجمالي الإنفاق (فقط للطلبات المكتملة)
            $total = $pdo->prepare("SELECT SUM(total_amount) FROM acrm_orders WHERE customer_phone = ? AND status = 'completed'");
            $total->execute([$phone]);
            $spent = $total->fetchColumn() ?: 0;
            
            $pdo->prepare("UPDATE acrm_customers SET total_spent = ? WHERE phone = ?")->execute([$spent, $phone]);
            
            echo json_encode(['status' => 'success']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['error' => true, 'message' => $e->getMessage()]);
    }
    exit;
}

// ============================================================
// 6. صفحة لوحة التحكم (Admin UI)
// ============================================================
function acrm_render_page_ultimate() {
    global $pdo;
    $tab = $_GET['tab'] ?? 'dashboard';
    
    $live = $pdo->query("SELECT COUNT(*) FROM acrm_sessions WHERE last_activity > DATE_SUB(NOW(), INTERVAL 10 MINUTE)")->fetchColumn() ?: 0;
    $sales = $pdo->query("SELECT SUM(total_amount) FROM acrm_orders WHERE status = 'completed'")->fetchColumn() ?: 0;
    $new_orders = $pdo->query("SELECT COUNT(*) FROM acrm_orders WHERE status = 'pending'")->fetchColumn() ?: 0;
    $cust_count = $pdo->query("SELECT COUNT(*) FROM acrm_customers")->fetchColumn() ?: 0;

    $chart_data = [];
    for($i=6; $i>=0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $day_total = $pdo->query("SELECT SUM(total_amount) FROM acrm_orders WHERE DATE(order_date) = '$date' AND status = 'completed'")->fetchColumn() ?: 0;
        $chart_data[] = ['date' => $date, 'total' => (float)$day_total];
    }
    $chart_labels = json_encode(array_column($chart_data, 'date'));
    $chart_values = json_encode(array_column($chart_data, 'total'));

    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>CRM الشامل - إدارة الطلبات</title>
        <script src="https://cdn.tailwindcss.com"></script>
        <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
        <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
        <style>body { font-family: 'Cairo', sans-serif; }</style>
    </head>
    <body class="bg-gray-50 flex h-screen overflow-hidden">

    <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl z-20">
        <div class="h-20 flex items-center justify-center border-b border-slate-800"><h2 class="text-2xl font-bold text-blue-400">إدارة المتجر</h2></div>
        <nav class="flex-1 px-4 py-6 space-y-2">
            <a href="index.php" class="flex items-center gap-3 px-4 py-3 text-slate-300 hover:bg-slate-800 rounded-xl transition">الرئيسية</a>
            <a href="plugins.php" class="flex items-center gap-3 px-4 py-3 bg-blue-600 text-white rounded-xl shadow-lg">الإضافات</a>
            
            <div class="pt-4 pb-2 text-xs font-bold text-slate-500 px-4 uppercase">التحليلات</div>
            <a href="?tool=analytics_crm_ultimate&tab=dashboard" class="flex items-center gap-3 px-4 py-2 <?= $tab=='dashboard'?'text-blue-400':'text-slate-400 hover:text-white' ?>">📊 لوحة المبيعات</a>
            <a href="?tool=analytics_crm_ultimate&tab=customers" class="flex items-center gap-3 px-4 py-2 <?= $tab=='customers'?'text-blue-400':'text-slate-400 hover:text-white' ?>">👥 الطلبات والعملاء</a>
            <a href="?tool=analytics_crm_ultimate&tab=products" class="flex items-center gap-3 px-4 py-2 <?= $tab=='products'?'text-blue-400':'text-slate-400 hover:text-white' ?>">🔥 المنتجات</a>
        </nav>
    </aside>

    <main class="flex-1 overflow-y-auto p-8 relative">
        
        <?php if($tab == 'dashboard'): ?>
            <h1 class="text-2xl font-bold text-gray-800 mb-6">نظرة عامة (الأرباح المحققة)</h1>
            
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-xs font-bold uppercase">طلبات جديدة (انتظار)</p>
                        <h3 class="text-3xl font-bold text-orange-500 mt-2"><?= $new_orders ?></h3>
                    </div>
                    <div class="w-10 h-10 bg-orange-100 rounded-full flex items-center justify-center text-orange-500">🔔</div>
                </div>
                <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-xs font-bold uppercase">صافي الأرباح (مكتمل)</p>
                        <h3 class="text-3xl font-bold text-emerald-600 mt-2"><?= number_format($sales) ?> ج.م</h3>
                    </div>
                    <div class="w-10 h-10 bg-emerald-100 rounded-full flex items-center justify-center text-emerald-600">💰</div>
                </div>
                <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-xs font-bold uppercase">إجمالي العملاء</p>
                        <h3 class="text-3xl font-bold text-blue-600 mt-2"><?= number_format($cust_count) ?></h3>
                    </div>
                    <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center text-blue-600">👥</div>
                </div>
                <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-gray-500 text-xs font-bold uppercase">زوار الآن</p>
                        <h3 class="text-3xl font-bold text-red-500 mt-2"><?= $live ?></h3>
                    </div>
                    <div class="w-10 h-10 bg-red-100 rounded-full flex items-center justify-center text-red-500">👁️</div>
                </div>
            </div>

            <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-100">
                <h3 class="text-lg font-bold text-gray-700 mb-4">نمو المبيعات المكتملة</h3>
                <div class="h-64">
                    <canvas id="salesChart"></canvas>
                </div>
            </div>
        <?php endif; ?>

        <?php if($tab == 'customers'): ?>
            <div class="flex justify-between items-center mb-6">
                <h1 class="text-2xl font-bold text-gray-800">إدارة الطلبات والعملاء</h1>
            </div>

            <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <table class="w-full text-right">
                    <thead class="bg-gray-50 text-gray-500 text-xs uppercase font-bold">
                        <tr>
                            <th class="p-4">العميل</th>
                            <th class="p-4">رقم الهاتف</th>
                            <th class="p-4">العنوان</th>
                            <th class="p-4">إجمالي الطلبات</th>
                            <th class="p-4">آخر نشاط</th>
                            <th class="p-4 text-center">إجراءات</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100 text-sm">
                        <?php 
                        $res = $pdo->query("SELECT * FROM acrm_customers ORDER BY last_order_date DESC LIMIT 50");
                        while($row = $res->fetch(PDO::FETCH_ASSOC)): ?>
                        <tr class="hover:bg-gray-50 transition">
                            <td class="p-4 font-bold text-gray-800"><?= htmlspecialchars($row['name']) ?></td>
                            <td class="p-4 font-mono text-emerald-600 dir-ltr text-left"><?= htmlspecialchars($row['phone']) ?></td>
                            <td class="p-4 text-gray-500 truncate max-w-xs"><?= htmlspecialchars($row['address']) ?></td>
                            <td class="p-4"><span class="bg-blue-100 text-blue-700 px-2 py-1 rounded text-xs font-bold"><?= $row['total_orders'] ?></span></td>
                            <td class="p-4 text-gray-400 text-xs"><?= $row['last_order_date'] ?></td>
                            <td class="p-4 text-center">
                                <button onclick="viewOrders('<?= htmlspecialchars($row['phone']) ?>', '<?= htmlspecialchars($row['name']) ?>')" 
                                    class="bg-slate-800 text-white px-3 py-1.5 rounded text-xs hover:bg-slate-700 transition flex items-center gap-1 mx-auto">
                                    <span>عرض الطلبات</span>
                                </button>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <?php if($tab == 'products'): 
            $top = $pdo->query("SELECT product_id, COUNT(*) as c FROM acrm_order_items GROUP BY product_id ORDER BY c DESC LIMIT 20")->fetchAll();
        ?>
            <h1 class="text-2xl font-bold text-gray-800 mb-6">تحليل المنتجات المطلوبة</h1>
            <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-6">
                <?php if($top): ?>
                    <?php foreach($top as $t): 
                        $name = $pdo->query("SELECT title FROM products WHERE id=".$t['product_id'])->fetchColumn();
                        if(!$name) {
                            $name = $pdo->query("SELECT product_name FROM acrm_order_items WHERE product_id=".$t['product_id']." LIMIT 1")->fetchColumn();
                        }
                    ?>
                    <div class="flex justify-between items-center border-b border-gray-50 py-3 last:border-0">
                        <div class="flex items-center gap-3">
                            <span class="w-8 h-8 rounded-full bg-indigo-50 text-indigo-600 flex items-center justify-center font-bold text-xs">#<?= $t['product_id'] ?></span>
                            <span class="font-bold text-gray-700"><?= $name ?></span>
                        </div>
                        <span class="bg-emerald-100 text-emerald-700 px-3 py-1 rounded-full text-xs font-bold"><?= $t['c'] ?> طلب</span>
                    </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="text-center py-10 text-gray-400">لا توجد بيانات كافية.</div>
                <?php endif; ?>
            </div>
        <?php endif; ?>

    </main>

    <div id="orderModal" class="fixed inset-0 bg-black/60 z-50 hidden flex items-center justify-center backdrop-blur-sm p-4">
        <div class="bg-white rounded-2xl shadow-2xl w-full max-w-3xl max-h-[90vh] flex flex-col overflow-hidden">
            <div class="p-5 border-b flex justify-between items-center bg-gray-50">
                <h3 class="font-bold text-lg text-gray-800">سجل طلبات العميل: <span id="modalCustName" class="text-blue-600"></span></h3>
                <button onclick="closeModal()" class="bg-gray-200 hover:bg-red-100 hover:text-red-500 rounded-full w-8 h-8 flex items-center justify-center transition">&times;</button>
            </div>
            <div id="modalContent" class="p-6 overflow-y-auto space-y-6 bg-slate-50/50 flex-1">
                <div class="text-center text-gray-400 py-10">جاري تحميل البيانات...</div>
            </div>
        </div>
    </div>

    <script>
        const ctx = document.getElementById('salesChart');
        if(ctx) {
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: <?= $chart_labels ?>,
                    datasets: [{
                        label: 'المبيعات المكتملة (ج.م)',
                        data: <?= $chart_values ?>,
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: { responsive: true, maintainAspectRatio: false }
            });
        }

        const modal = document.getElementById('orderModal');
        const content = document.getElementById('modalContent');
        let currentPhone = '';

        function viewOrders(phone, name) {
            currentPhone = phone;
            document.getElementById('modalCustName').innerText = name;
            modal.classList.remove('hidden');
            content.innerHTML = '<div class="text-center py-10"><div class="animate-spin w-8 h-8 border-4 border-blue-600 border-t-transparent rounded-full mx-auto mb-2"></div>جاري جلب البيانات...</div>';

            fetch('?tool=analytics_crm_ultimate&ajax_type=get_customer_orders&phone=' + phone)
                .then(res => res.json())
                .then(data => {
                    if(data.error) throw new Error(data.message);
                    if(data.length === 0) {
                        content.innerHTML = '<div class="text-center text-gray-500 py-10 bg-white rounded-xl border border-dashed">لا توجد طلبات مسجلة لهذا العميل.</div>';
                        return;
                    }
                    
                    let html = '';
                    
                    // ترجمة طرق الدفع
                    const payMap = {
                        'cod': 'دفع عند الاستلام 🏠',
                        'wallet': 'محفظة إلكترونية 📱',
                        'instapay': 'InstaPay 🏦'
                    };

                    data.forEach(order => {
                        let statusColor = 'bg-yellow-100 text-yellow-700 border-yellow-200';
                        let statusText = 'قيد الانتظار (جديد)';
                        if(order.status === 'completed') { statusColor = 'bg-green-100 text-green-700 border-green-200'; statusText = 'مكتمل (تم الدفع)'; } 
                        else if(order.status === 'cancelled') { statusColor = 'bg-red-100 text-red-700 border-red-200'; statusText = 'ملغي'; }

                        // تحديد اسم طريقة الدفع
                        let payMethodName = payMap[order.payment_method] || order.payment_method || 'غير محدد';

                        let itemsHtml = order.items.map(item => {
                            // *** التصحيح الجذري لمسار الصورة ***
                            let imgSrc = 'https://via.placeholder.com/100x100?text=No+Img';
                            
                            if (item.item_image && item.item_image !== 'null') {
                                if (item.item_image.startsWith('http')) {
                                    imgSrc = item.item_image;
                                } else if (item.item_image.indexOf('assets/uploads/') !== -1) {
                                    imgSrc = '../' + item.item_image;
                                } else {
                                    imgSrc = '../assets/uploads/' + item.item_image;
                                }
                            }
                            
                            let colorHtml = '';
                            if(item.item_color && item.item_color !== 'null') {
                                colorHtml = `<div class="flex items-center gap-1 text-xs text-gray-500 mt-1"><span class="w-4 h-4 rounded-full border border-gray-300 shadow-sm" style="background-color: ${item.item_color}"></span> <span>اللون المحدد</span></div>`;
                            }

                            let sizeHtml = '';
                            if(item.item_size && item.item_size !== 'null') {
                                sizeHtml = `<span class="text-xs bg-gray-100 px-2 py-0.5 rounded border text-gray-600">المقاس: ${item.item_size}</span>`;
                            }

                            return `
                            <div class="flex items-start gap-4 p-3 bg-gray-50 rounded-xl border border-gray-100 mb-2 last:mb-0">
                                <div class="w-16 h-16 flex-shrink-0 bg-white rounded-lg border border-gray-200 overflow-hidden">
                                    <img src="${imgSrc}" class="w-full h-full object-cover" onerror="this.src='https://via.placeholder.com/100x100?text=Err'">
                                </div>
                                <div class="flex-1">
                                    <div class="font-bold text-gray-800 text-sm mb-1">${item.product_name}</div>
                                    <div class="flex flex-wrap gap-2 items-center">
                                        ${sizeHtml}
                                        ${colorHtml}
                                    </div>
                                    <div class="text-xs text-gray-400 mt-1">سعر الوحدة: ${parseFloat(item.price).toLocaleString()} ج.م</div>
                                </div>
                                <div class="text-right">
                                    <div class="font-bold text-blue-600">x${item.quantity}</div>
                                    <div class="text-xs font-bold text-gray-700 mt-1">
                                        ${(item.price * item.quantity).toLocaleString()} ج.م
                                    </div>
                                </div>
                            </div>
                            `;
                        }).join('');

                        html += `
                        <div class="bg-white border border-gray-200 rounded-2xl overflow-hidden shadow-sm hover:shadow-md transition">
                            <div class="p-4 bg-gray-50 border-b flex flex-wrap justify-between items-center gap-4">
                                <div class="flex items-center gap-3">
                                    <div class="text-sm">
                                        <div class="font-bold text-gray-800 flex items-center gap-2">
                                            طلب #${order.id} 
                                            <span class="text-xs font-normal text-gray-500 bg-gray-100 px-2 py-0.5 rounded border">
                                                ${payMethodName}
                                            </span>
                                        </div>
                                        <div class="text-xs text-gray-500 mt-1">${order.order_date}</div>
                                    </div>
                                    <span class="px-3 py-1 rounded-full text-xs font-bold border ${statusColor}">${statusText}</span>
                                </div>
                                <div class="flex items-center gap-3">
                                    <div class="font-bold text-lg text-emerald-700">${parseFloat(order.total_amount).toLocaleString()} ج.م</div>
                                    <select onchange="updateStatus(${order.id}, this.value)" class="text-xs border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-blue-500 bg-white py-1 px-2 cursor-pointer">
                                        <option value="pending" ${order.status == 'pending' ? 'selected' : ''}>قيد الانتظار</option>
                                        <option value="completed" ${order.status == 'completed' ? 'selected' : ''}>مكتمل (أرباح)</option>
                                        <option value="cancelled" ${order.status == 'cancelled' ? 'selected' : ''}>ملغي</option>
                                    </select>
                                </div>
                            </div>
                            <div class="p-4">${itemsHtml}</div>
                        </div>
                        `;
                    });
                    content.innerHTML = html;
                })
                .catch(err => {
                    content.innerHTML = `<div class="text-center text-red-500 p-4 border border-red-200 bg-red-50 rounded-xl">حدث خطأ: ${err.message}</div>`;
                });
        }

        function updateStatus(orderId, newStatus) {
            if(!confirm('هل أنت متأكد من تغيير حالة الطلب؟ (سيؤثر هذا على حساب الأرباح)')) return;
            const select = event.target;
            select.disabled = true;
            let formData = new FormData();
            formData.append('order_id', orderId);
            formData.append('status', newStatus);
            formData.append('phone', currentPhone);

            fetch('?tool=analytics_crm_ultimate&ajax_type=update_order_status', { method: 'POST', body: formData })
            .then(r => r.json())
            .then(data => {
                select.disabled = false;
                if(data.status === 'success') {
                    viewOrders(currentPhone, document.getElementById('modalCustName').innerText);
                } else { alert('فشل التحديث'); }
            });
        }

        function closeModal() { modal.classList.add('hidden'); }
        modal.addEventListener('click', (e) => { if (e.target === modal) closeModal(); });
    </script>
    </body>
    </html>
    <?php
}
?>