<?php
/*
Plugin Name: مدير الموضة (Fashion Pro)
Plugin Key: fashion_pro
Description: إضافة المقاسات والألوان للمنتجات مع واجهة اختيار متطورة.
*/

// ============================================================
// 1. تجهيز قاعدة البيانات (إضافة عمود الخيارات)
// ============================================================
global $pdo;
if ($pdo) {
    // نتأكد إن الجدول فيه عمود للخيارات
    try {
        $pdo->query("SELECT variants FROM products LIMIT 1");
    } catch (Exception $e) {
        $pdo->exec("ALTER TABLE products ADD COLUMN variants TEXT DEFAULT NULL");
    }
}

// ============================================================
// 2. زر القائمة الجانبية
// ============================================================
function fashion_sidebar_link() {
    $active = (isset($_GET['tool']) && $_GET['tool'] == 'fashion_pro') ? 'bg-emerald-600 text-white shadow-lg' : 'text-slate-300 hover:bg-slate-800 hover:text-white';
    echo '<a href="plugins.php?tool=fashion_pro" class="flex items-center gap-3 px-4 py-3 rounded-xl transition '.$active.'">
            <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21a4 4 0 01-4-4V5a2 2 0 012-2h4a2 2 0 012 2v12a4 4 0 01-4 4zm0 0h12a2 2 0 002-2v-4a2 2 0 00-2-2h-2.343M11 7.343l1.657-1.657a2 2 0 012.828 0l2.829 2.829a2 2 0 010 2.828l-8.486 8.485M7 17h.01" /></svg>
            <span>إدارة المقاسات والألوان</span>
          </a>';
}
add_action('admin_sidebar', 'fashion_sidebar_link');

// ============================================================
// 3. صفحة الإدارة (تحديد المقاسات والألوان لكل منتج)
// ============================================================
if (isset($_GET['tool']) && $_GET['tool'] == 'fashion_pro') {
    fashion_render_admin();
    exit;
}

function fashion_render_admin() {
    global $pdo;
    $msg = '';
    
    // حفظ البيانات
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['product_id'])) {
        $pid = $_POST['product_id'];
        $sizes = isset($_POST['sizes']) ? json_encode($_POST['sizes']) : '[]';
        $colors = isset($_POST['colors']) ? json_encode($_POST['colors']) : '[]';
        
        $data = json_encode(['sizes' => json_decode($sizes), 'colors' => json_decode($colors)]);
        
        $stmt = $pdo->prepare("UPDATE products SET variants = ? WHERE id = ?");
        $stmt->execute([$data, $pid]);
        $msg = "تم تحديث خيارات المنتج بنجاح ✅";
    }

    $products = $pdo->query("SELECT id, title, image, variants FROM products ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>إدارة الموضة</title>
        <script src="https://cdn.tailwindcss.com"></script>
        <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
        <style>body { font-family: 'Cairo', sans-serif; }</style>
    </head>
    <body class="bg-gray-50 flex h-screen overflow-hidden">
        <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl">
            <div class="h-20 flex items-center justify-center border-b border-slate-800"><h2 class="text-2xl font-bold text-emerald-400">WhatsCart</h2></div>
            <nav class="flex-1 px-4 py-6 space-y-2">
                <a href="plugins.php" class="flex items-center gap-3 px-4 py-3 bg-emerald-600 text-white rounded-xl shadow-lg">رجوع للإضافات</a>
            </nav>
        </aside>

        <main class="flex-1 overflow-y-auto p-8">
            <h1 class="text-2xl font-bold text-gray-800 mb-6">👕 إدارة المقاسات والألوان</h1>
            <?php if($msg): ?><div class="bg-green-100 text-green-700 p-4 rounded-xl mb-4 font-bold"><?= $msg ?></div><?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                <?php foreach($products as $p): 
                    $v = json_decode($p['variants'] ?? '{"sizes":[], "colors":[]}', true);
                    $s = $v['sizes'] ?? [];
                    $c = $v['colors'] ?? [];
                ?>
                <div class="bg-white p-5 rounded-xl shadow-sm border border-gray-100">
                    <div class="flex items-center gap-3 mb-4">
                        <img src="../assets/uploads/<?= $p['image'] ?>" class="w-12 h-12 rounded object-cover">
                        <h3 class="font-bold text-gray-800 text-sm"><?= $p['title'] ?></h3>
                    </div>
                    
                    <form method="POST">
                        <input type="hidden" name="product_id" value="<?= $p['id'] ?>">
                        
                        <div class="mb-3">
                            <label class="block text-xs font-bold text-gray-500 mb-1">المقاسات:</label>
                            <div class="flex flex-wrap gap-2">
                                <?php foreach(['S','M','L','XL','2XL','3XL','40','41','42','43','44','45'] as $size): ?>
                                <label class="cursor-pointer">
                                    <input type="checkbox" name="sizes[]" value="<?= $size ?>" <?= in_array($size, $s) ? 'checked' : '' ?> class="peer sr-only">
                                    <span class="px-2 py-1 text-xs border rounded peer-checked:bg-slate-800 peer-checked:text-white transition hover:bg-gray-50"><?= $size ?></span>
                                </label>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <div class="mb-4">
                            <label class="block text-xs font-bold text-gray-500 mb-1">الألوان:</label>
                            <div class="flex flex-wrap gap-2">
                                <?php 
                                $colors_map = ['الأحمر'=>'bg-red-500','الأزرق'=>'bg-blue-500','الأسود'=>'bg-black','الأبيض'=>'bg-white border','الأخضر'=>'bg-green-500','الأصفر'=>'bg-yellow-400','الرمادي'=>'bg-gray-500','البيج'=>'bg-[#f5f5dc] border'];
                                foreach($colors_map as $name => $class): 
                                ?>
                                <label class="cursor-pointer" title="<?= $name ?>">
                                    <input type="checkbox" name="colors[]" value="<?= $name ?>" <?= in_array($name, $c) ? 'checked' : '' ?> class="peer sr-only">
                                    <div class="w-6 h-6 rounded-full <?= $class ?> peer-checked:ring-2 ring-offset-1 ring-emerald-500 shadow-sm"></div>
                                </label>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <button class="w-full bg-emerald-600 text-white py-2 rounded-lg text-sm font-bold hover:bg-emerald-700">حفظ الخيارات</button>
                    </form>
                </div>
                <?php endforeach; ?>
            </div>
        </main>
    </body>
    </html>
    <?php
}

// ============================================================
// 4. الواجهة الأمامية (إظهار الأزرار للعميل)
// ============================================================
function fashion_frontend_ui($product) {
    if (empty($product['variants'])) return;
    
    $v = json_decode($product['variants'], true);
    $sizes = $v['sizes'] ?? [];
    $colors = $v['colors'] ?? [];
    
    if (empty($sizes) && empty($colors)) return;
    
    echo '<div class="mt-3 product-options" data-id="'.$product['id'].'">';
    
    // عرض المقاسات
    if (!empty($sizes)) {
        echo '<div class="mb-2">
                <span class="text-xs text-gray-500 font-bold block mb-1">المقاس:</span>
                <div class="flex flex-wrap gap-1">';
        foreach ($sizes as $size) {
            echo '<button onclick="selectOption(this, \'size\', '.$product['id'].')" class="opt-btn size-btn border px-2 py-1 rounded text-xs hover:border-black transition" data-val="'.$size.'">'.$size.'</button>';
        }
        echo '</div><input type="hidden" id="selected_size_'.$product['id'].'"></div>';
    }

    // عرض الألوان
    if (!empty($colors)) {
        $colors_map = ['الأحمر'=>'bg-red-500','الأزرق'=>'bg-blue-500','الأسود'=>'bg-black','الأبيض'=>'bg-white border','الأخضر'=>'bg-green-500','الأصفر'=>'bg-yellow-400','الرمادي'=>'bg-gray-500','البيج'=>'bg-[#f5f5dc] border'];
        echo '<div>
                <span class="text-xs text-gray-500 font-bold block mb-1">اللون:</span>
                <div class="flex flex-wrap gap-1">';
        foreach ($colors as $color) {
            $bg = $colors_map[$color] ?? 'bg-gray-200';
            echo '<button onclick="selectOption(this, \'color\', '.$product['id'].')" class="opt-btn color-btn w-6 h-6 rounded-full '.$bg.' shadow-sm border border-transparent hover:scale-110 transition" data-val="'.$color.'" title="'.$color.'"></button>';
        }
        echo '</div><input type="hidden" id="selected_color_'.$product['id'].'"></div>';
    }
    
    echo '</div>';
}
add_action('after_product_title', 'fashion_frontend_ui');


// ============================================================
// 5. الجافاسكريبت (المنطق السحري لدمج الخيارات في الطلب)
// ============================================================
function fashion_footer_scripts() {
    ?>
    <script>
    // 1. دالة اختيار الأزرار وتلوينها
    function selectOption(btn, type, pid) {
        // إزالة التحديد من الأخوة
        let parent = btn.parentElement;
        let siblings = parent.getElementsByClassName('opt-btn');
        for(let sib of siblings) {
            sib.classList.remove('ring-2', 'ring-black', 'bg-slate-800', 'text-white', 'border-black');
            if(type === 'size') sib.classList.remove('bg-slate-800', 'text-white'); 
        }
        
        // تحديد الزر الحالي
        if(type === 'color') btn.classList.add('ring-2', 'ring-offset-1', 'ring-black');
        if(type === 'size') btn.classList.add('bg-slate-800', 'text-white', 'border-black');
        
        // حفظ القيمة في المدخل المخفي
        document.getElementById('selected_' + type + '_' + pid).value = btn.getAttribute('data-val');
    }

    // 2. تعديل دالة الإضافة للسلة الأصلية (Override)
    // بنستنى شوية عشان نتأكد إن Alpine حمل
    document.addEventListener('alpine:init', () => {
        const originalApp = window.app;
        
        window.app = function() {
            const component = originalApp();
            const originalAdd = component.add;

            // استبدال دالة الإضافة
            component.add = function(id, title, price) {
                // البحث عن خيارات المنتج
                let sizeInput = document.getElementById('selected_size_' + id);
                let colorInput = document.getElementById('selected_color_' + id);
                
                let selectedSize = sizeInput ? sizeInput.value : null;
                let selectedColor = colorInput ? colorInput.value : null;
                
                // التحقق من الإجبار (لو المنتج ليه خيارات لازم يختارها)
                let container = document.querySelector(`.product-options[data-id="${id}"]`);
                if (container) {
                    if (sizeInput && !selectedSize) {
                        alert('⚠️ يرجى اختيار المقاس أولاً');
                        return;
                    }
                    if (colorInput && !selectedColor) {
                        alert('⚠️ يرجى اختيار اللون أولاً');
                        return;
                    }
                }

                // تجهيز الاسم الجديد (تيشرت - أحمر - XL)
                let newTitle = title;
                let variantKey = "";
                
                if (selectedSize || selectedColor) {
                    let details = [];
                    if (selectedColor) details.push(selectedColor);
                    if (selectedSize) details.push(selectedSize);
                    let variantString = details.join(' / ');
                    
                    newTitle = `${title} [${variantString}]`;
                    variantKey = variantString; // عشان نميز المنتجات في السلة
                }

                // هنا الحيلة: بنغير الـ ID عشان السلة تعتبره منتج مختلف
                // مثلاً المنتج رقم 5 أحمر غير المنتج رقم 5 أزرق
                let uniqueId = id; 
                if (variantKey) {
                    uniqueId = id + '_' + variantKey.replace(/\s+/g, '');
                }

                // الإضافة للسلة اليدوية (لأن الدالة الأصلية مش هتقبل الـ ID النصي بسهولة)
                let item = this.cart.find(i => i.id === uniqueId);
                if (item) {
                    item.qty++;
                } else {
                    this.cart.push({ id: uniqueId, title: newTitle, price: price, qty: 1 });
                }
                
                this.save();
                this.cartOpen = true;
                
                // تصفير الاختيارات بعد الإضافة (اختياري)
                // if(sizeInput) sizeInput.value = '';
                // if(colorInput) colorInput.value = '';
                // إزالة كلاسات التحديد بصرياً... (ممكن نسيبها عشان يشتري تاني بسهولة)
            };

            return component;
        };
    });
    </script>
    <style>
        .opt-btn { transition: all 0.2s; }
    </style>
    <?php
}
add_action('site_footer', 'fashion_footer_scripts');
?>