<?php
/*
Plugin Name: مدير النسخ الاحتياطي (شامل)
Plugin Key: products_backup
Description: تصدير واستيراد المنتجات بالكامل (الصور + البيانات) ويدعم ملفات JSON و ZIP.
*/

// ============================================================
// 1. زر القائمة الجانبية
// ============================================================
function backup_sidebar_link() {
    $active = (isset($_GET['tool']) && $_GET['tool'] == 'products_backup') ? 'bg-emerald-600 text-white shadow-lg' : 'text-slate-300 hover:bg-slate-800 hover:text-white';
    echo '<a href="plugins.php?tool=products_backup" class="flex items-center gap-3 px-4 py-3 rounded-xl transition '.$active.'">
            <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7v8a2 2 0 002 2h6M8 7V5a2 2 0 012-2h4.586a1 1 0 01.707.293l4.414 4.414a1 1 0 01.293.707V15a2 2 0 01-2 2h-2M8 7H6a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2v-2" /></svg>
            <span>نسخة احتياطية كاملة</span>
          </a>';
}
add_action('admin_sidebar', 'backup_sidebar_link');

// ============================================================
// 2. المحرك والمنطق (Backend Logic)
// ============================================================
if (isset($_GET['tool']) && $_GET['tool'] == 'products_backup') {
    
    // رفع حدود الذاكرة والوقت للتعامل مع الصور الكبيرة
    ini_set('memory_limit', '512M');
    set_time_limit(600); 

    // --- (أ) التصدير (Export ZIP) ---
    if (isset($_GET['action']) && $_GET['action'] == 'download') {
        global $pdo;
        
        $products = $pdo->query("SELECT * FROM products")->fetchAll(PDO::FETCH_ASSOC);
        $json_data = json_encode($products, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        $zip = new ZipArchive();
        $zipName = 'Full_Backup_' . date('Y-m-d_H-i') . '.zip';
        $tmpFile = tempnam(sys_get_temp_dir(), 'zip');
        
        if ($zip->open($tmpFile, ZipArchive::CREATE) === TRUE) {
            // إضافة ملف البيانات
            $zip->addFromString('data.json', $json_data);
            
            // إضافة الصور الموجودة فعلياً
            $uploadDir = "../assets/uploads/";
            $countImg = 0;
            foreach ($products as $p) {
                if (!empty($p['image']) && file_exists($uploadDir . $p['image'])) {
                    $zip->addFile($uploadDir . $p['image'], 'images/' . $p['image']);
                    $countImg++;
                }
            }
            $zip->setArchiveComment("WhatsCart Backup: " . count($products) . " Products, " . $countImg . " Images.");
            $zip->close();
            
            // تحميل الملف
            header('Content-Type: application/zip');
            header('Content-disposition: attachment; filename='.$zipName);
            header('Content-Length: ' . filesize($tmpFile));
            readfile($tmpFile);
            unlink($tmpFile);
            exit;
        } else {
            die("خطأ: لا يمكن إنشاء ملف ZIP. تأكد من تفعيل مكتبة ZipArchive في السيرفر.");
        }
    }

    // --- (ب) الاستيراد (Import ZIP or JSON) ---
    $msg = '';
    $error = '';
    
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['backup_file'])) {
        if ($_FILES['backup_file']['error'] == 0) {
            $fileVal = $_FILES['backup_file'];
            $ext = strtolower(pathinfo($fileVal['name'], PATHINFO_EXTENSION));
            
            // 1. التعامل مع ملف ZIP (بيانات + صور)
            if ($ext == 'zip') {
                $zip = new ZipArchive();
                if ($zip->open($fileVal['tmp_name']) === TRUE) {
                    $jsonContent = $zip->getFromName('data.json');
                    if ($jsonContent) {
                        $data = json_decode($jsonContent, true);
                        if (is_array($data)) {
                            $imported = import_products_to_db($data); // دالة الإدخال
                            
                            // استخراج الصور
                            $targetDir = "../assets/uploads/";
                            if (!is_dir($targetDir)) mkdir($targetDir, 0755, true);
                            
                            $imgCount = 0;
                            for($i = 0; $i < $zip->numFiles; $i++){
                                $filename = $zip->getNameIndex($i);
                                // التأكد أن الملف صورة وموجود داخل مجلد images/
                                if(strpos($filename, 'images/') === 0 && $filename != 'images/') {
                                    $baseName = basename($filename);
                                    if(!empty($baseName)) {
                                        copy("zip://".$fileVal['tmp_name']."#".$filename, $targetDir . $baseName);
                                        $imgCount++;
                                    }
                                }
                            }
                            $msg = "تم استعادة $imported منتج و $imgCount صورة بنجاح! 📦";
                        } else { $error = "ملف البيانات داخل الأرشيف تالف."; }
                    } else { $error = "الملف data.json مفقود داخل ملف الـ ZIP."; }
                    $zip->close();
                } else { $error = "فشل فتح ملف الـ ZIP."; }
            
            // 2. التعامل مع ملف JSON (بيانات فقط - الطريقة القديمة)
            } elseif ($ext == 'json') {
                $content = file_get_contents($fileVal['tmp_name']);
                $data = json_decode($content, true);
                if (is_array($data)) {
                    $imported = import_products_to_db($data);
                    $msg = "تم استعادة $imported منتج بنجاح (بيانات فقط) 📄.";
                } else { $error = "ملف JSON غير صالح."; }
            } else {
                $error = "يرجى رفع ملف ZIP أو JSON.";
            }
        } else {
            $error = "حدث خطأ أثناء رفع الملف.";
        }
    }

    backup_render_ui_v2($msg, $error);
    exit;
}

// دالة مساعدة لإدخال المنتجات للداتا بيز
function import_products_to_db($data) {
    global $pdo;
    $count = 0;
    // استعلام الإدخال (نتجاهل الـ ID القديم ليأخذ ID جديد، أو يمكن تعديله للتحديث)
    $stmt = $pdo->prepare("INSERT INTO products (title, price, description, image, active, category_id, variants) VALUES (?, ?, ?, ?, ?, ?, ?)");
    
    foreach ($data as $item) {
        if (!empty($item['title'])) {
            $stmt->execute([
                $item['title'],
                $item['price'],
                $item['description'] ?? '',
                $item['image'] ?? 'default.jpg',
                $item['active'] ?? 1,
                $item['category_id'] ?? 0,
                $item['variants'] ?? null
            ]);
            $count++;
        }
    }
    return $count;
}

// ============================================================
// 3. واجهة المستخدم (UI) - بتصميم كامل
// ============================================================
function backup_render_ui_v2($msg, $error) {
    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>النسخ الاحتياطي الشامل</title>
        <script src="https://cdn.tailwindcss.com"></script>
        <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
        <style>body { font-family: 'Cairo', sans-serif; }</style>
    </head>
    <body class="bg-gray-50 flex h-screen overflow-hidden">
        
        <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl">
            <div class="h-20 flex items-center justify-center border-b border-slate-800"><h2 class="text-2xl font-bold text-emerald-400">WhatsCart</h2></div>
            <nav class="flex-1 px-4 py-6 space-y-2">
                <a href="plugins.php" class="flex items-center gap-3 px-4 py-3 bg-emerald-600 text-white rounded-xl shadow-lg">رجوع للإضافات</a>
            </nav>
        </aside>

        <main class="flex-1 overflow-y-auto p-8">
            <div class="flex justify-between items-center mb-8">
                <div>
                    <h1 class="text-2xl font-bold text-gray-800 flex items-center gap-2">
                        <span class="text-3xl">📦</span> النسخ الاحتياطي (صور + بيانات)
                    </h1>
                    <p class="text-gray-500 mt-1">يمكنك حفظ نسخة كاملة من متجرك واستعادتها في أي وقت.</p>
                </div>
            </div>

            <?php if($msg): ?>
                <div class="bg-green-50 border border-green-200 text-green-700 px-6 py-4 rounded-xl mb-6 font-bold flex items-center gap-3 shadow-sm">
                    <span class="bg-green-200 p-1 rounded-full text-green-800">✓</span> <?= $msg ?>
                </div>
            <?php endif; ?>
            
            <?php if($error): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-6 py-4 rounded-xl mb-6 font-bold flex items-center gap-3 shadow-sm">
                    <span class="bg-red-200 p-1 rounded-full text-red-800">✕</span> <?= $error ?>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden relative group">
                    <div class="absolute top-0 right-0 w-2 h-full bg-indigo-500"></div>
                    <div class="p-8">
                        <div class="flex items-center gap-4 mb-6">
                            <div class="bg-indigo-50 p-3 rounded-2xl text-indigo-600">
                                <svg class="w-8 h-8" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" /></svg>
                            </div>
                            <div>
                                <h3 class="text-xl font-bold text-gray-800">تصدير نسخة كاملة</h3>
                                <p class="text-xs text-gray-400 font-mono mt-1">Full Backup (.zip)</p>
                            </div>
                        </div>
                        <p class="text-gray-600 mb-8 leading-relaxed">
                            سيتم إنشاء ملف مضغوط يحتوي على:
                            <ul class="list-disc list-inside mt-2 text-sm text-gray-500">
                                <li>قاعدة البيانات (المنتجات، الأقسام، الأسعار).</li>
                                <li>مجلد الصور بالكامل (Images).</li>
                            </ul>
                        </p>
                        <a href="plugins.php?tool=products_backup&action=download" class="block w-full bg-indigo-600 text-white text-center py-4 rounded-xl font-bold hover:bg-indigo-700 transition shadow-lg hover:shadow-indigo-200 transform active:scale-95">
                            بدء التحميل الآن 🚀
                        </a>
                    </div>
                </div>

                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden relative group">
                    <div class="absolute top-0 right-0 w-2 h-full bg-emerald-500"></div>
                    <div class="p-8">
                        <div class="flex items-center gap-4 mb-6">
                            <div class="bg-emerald-50 p-3 rounded-2xl text-emerald-600">
                                <svg class="w-8 h-8" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12" /></svg>
                            </div>
                            <div>
                                <h3 class="text-xl font-bold text-gray-800">استعادة نسخة</h3>
                                <p class="text-xs text-gray-400 font-mono mt-1">Supports (.zip) & (.json)</p>
                            </div>
                        </div>
                        
                        <form method="POST" enctype="multipart/form-data">
                            <div class="border-2 border-dashed border-gray-200 rounded-xl p-6 text-center mb-6 bg-gray-50 hover:bg-white hover:border-emerald-400 transition cursor-pointer relative">
                                <input type="file" name="backup_file" accept=".zip,.json" class="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10" onchange="document.getElementById('fileName').innerText = this.files[0].name">
                                <svg class="w-10 h-10 mx-auto text-gray-300 mb-2" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/></svg>
                                <p class="text-sm text-gray-500 font-bold" id="fileName">اضغط لاختيار ملف النسخة الاحتياطية</p>
                                <p class="text-xs text-gray-400 mt-1">يقبل ZIP (كامل) أو JSON (بيانات فقط)</p>
                            </div>
                            
                            <button type="submit" class="block w-full bg-emerald-600 text-white text-center py-4 rounded-xl font-bold hover:bg-emerald-700 transition shadow-lg hover:shadow-emerald-200 transform active:scale-95">
                                استعادة البيانات 🔄
                            </button>
                        </form>
                    </div>
                </div>

            </div>
        </main>
    </div>
    </body>
    </html>
    <?php
}
?>