<?php
/*
Plugin Name: الفئات والبحث الذكي (Smart Search & Cats)
Plugin Key: smart_cats_search
Description: الإصدار المصحح - حل مشكلة عدم ظهور المنتجات عند اختيار الفئة (Type Mismatch Fix).
Version: 2.2
*/

// ============================================================
// 1. إعداد قاعدة البيانات (يدعم العربية UTF8mb4)
// ============================================================
global $pdo;
if ($pdo) {
    // إجبار الاتصال على استخدام UTF-8
    try {
        $pdo->exec("SET NAMES utf8mb4");
    } catch (Exception $e) {}

    // 1. إنشاء جدول الفئات
    try {
        $pdo->exec("CREATE TABLE IF NOT EXISTS `smart_categories` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `name` varchar(255) NOT NULL,
            `color` varchar(20) DEFAULT 'bg-gray-100',
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // إصلاح ترميز الجدول القديم إن وجد
        $pdo->exec("ALTER TABLE `smart_categories` CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
    } catch (Exception $e) {}

    // 2. إضافة عمود الفئات لجدول المنتجات
    try {
        $pdo->query("SELECT category_ids FROM products LIMIT 1");
    } catch (Exception $e) {
        try {
            $pdo->exec("ALTER TABLE products ADD COLUMN category_ids TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci DEFAULT NULL");
        } catch (Exception $ex) {}
    }
}

// ============================================================
// 2. زر القائمة الجانبية
// ============================================================
function smart_cats_sidebar_link() {
    $active = (isset($_GET['tool']) && $_GET['tool'] == 'smart_cats_search') ? 'bg-emerald-600 text-white shadow-lg' : 'text-slate-300 hover:bg-slate-800 hover:text-white';
    echo '<a href="plugins.php?tool=smart_cats_search" class="flex items-center gap-3 px-4 py-3 rounded-xl transition '.$active.'">
            <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" /></svg>
            <span>إدارة الفئات والبحث</span>
          </a>';
}
add_action('admin_sidebar', 'smart_cats_sidebar_link');

// ============================================================
// 3. صفحة الإدارة (Admin Logic)
// ============================================================
if (isset($_GET['tool']) && $_GET['tool'] == 'smart_cats_search') {
    smart_cats_render_admin();
    exit;
}

function smart_cats_render_admin() {
    global $pdo;
    $msg = '';
    
    // أ. حفظ الفئات للمنتج
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['update_cats'])) {
        $pid = $_POST['product_id'];
        $cats = isset($_POST['cats']) ? $_POST['cats'] : [];
        
        // [تعديل هام]: تحويل القيم إلى أرقام صحيحة لضمان التوافق
        $cats = array_map('intval', $cats);
        
        $json = json_encode($cats, JSON_UNESCAPED_UNICODE);
        $stmt = $pdo->prepare("UPDATE products SET category_ids = ? WHERE id = ?");
        $stmt->execute([$json, $pid]);
        $msg = "تم تحديث فئات المنتج بنجاح ✅";
    }

    // ب. إضافة فئة جديدة
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_cat'])) {
        $name = trim($_POST['cat_name']);
        if ($name) {
            $stmt = $pdo->prepare("INSERT INTO smart_categories (name) VALUES (?)");
            $stmt->execute([$name]);
            $msg = "تمت إضافة الفئة بنجاح ✅";
        }
    }

    // ج. حذف فئة
    if (isset($_GET['delete_cat'])) {
        $did = intval($_GET['delete_cat']);
        $pdo->prepare("DELETE FROM smart_categories WHERE id = ?")->execute([$did]);
        echo "<script>window.location.href='plugins.php?tool=smart_cats_search';</script>";
    }

    // جلب البيانات
    $categories = $pdo->query("SELECT * FROM smart_categories ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
    $products = $pdo->query("SELECT id, title, image, category_ids FROM products ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <title>إدارة الفئات</title>
        <script src="https://cdn.tailwindcss.com"></script>
        <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
        <style>body { font-family: 'Cairo', sans-serif; }</style>
    </head>
    <body class="bg-gray-50 flex h-screen overflow-hidden">
        
        <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl">
            <div class="h-20 flex items-center justify-center border-b border-slate-800">
                <h2 class="text-2xl font-bold text-emerald-400">WhatsCart</h2>
            </div>
            <nav class="flex-1 px-4 py-6 space-y-2">
                <a href="plugins.php" class="flex items-center gap-3 px-4 py-3 bg-emerald-600 text-white rounded-xl shadow-lg">رجوع للإضافات</a>
            </nav>
        </aside>

        <main class="flex-1 overflow-y-auto p-8">
            <h1 class="text-2xl font-bold text-gray-800 mb-6">🏷️ إدارة الفئات والبحث</h1>
            <?php if($msg): ?><div class="bg-green-100 text-green-700 p-4 rounded-xl mb-6 font-bold text-center shadow-sm border border-green-200"><?= $msg ?></div><?php endif; ?>

            <div class="bg-white p-4 rounded-xl shadow-sm border border-gray-100 mb-8 flex gap-4 items-end sticky top-0 z-10">
                <div class="flex-1">
                    <label class="block text-xs font-bold text-gray-500 mb-1">إضافة فئة جديدة</label>
                    <input type="text" name="cat_name" form="cat_form" class="w-full border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-emerald-500 outline-none" placeholder="مثال: رجالي، إلكترونيات..." required>
                </div>
                <form method="POST" id="cat_form" class="mb-0">
                    <button type="submit" name="add_cat" class="bg-slate-800 text-white px-6 py-2 rounded-lg font-bold hover:bg-slate-700 h-[42px]">إضافة فئة</button>
                </form>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                <?php foreach($products as $p): 
                    $p_cats = json_decode($p['category_ids'] ?? '[]', true);
                    $p_cats = is_array($p_cats) ? $p_cats : [];
                ?>
                <div class="bg-white p-5 rounded-xl shadow-sm border border-gray-100 hover:shadow-md transition">
                    <div class="flex items-center gap-3 mb-4">
                        <img src="../assets/uploads/<?= $p['image'] ?>" class="w-14 h-14 rounded-lg object-cover bg-gray-100 border">
                        <div class="truncate">
                            <h3 class="font-bold text-gray-800 text-sm leading-tight"><?= $p['title'] ?></h3>
                            <span class="text-xs text-gray-400">ID: <?= $p['id'] ?></span>
                        </div>
                    </div>
                    
                    <form method="POST">
                        <input type="hidden" name="product_id" value="<?= $p['id'] ?>">
                        <div class="mb-4">
                            <label class="block text-xs font-bold text-gray-500 mb-2">الفئات:</label>
                            <div class="flex flex-wrap gap-2 max-h-32 overflow-y-auto custom-scroll">
                                <?php if(empty($categories)): ?>
                                    <span class="text-xs text-gray-400">لا توجد فئات.</span>
                                <?php else: ?>
                                    <?php foreach($categories as $c): ?>
                                    <label class="cursor-pointer group">
                                        <input type="checkbox" name="cats[]" value="<?= $c['id'] ?>" <?= in_array($c['id'], $p_cats) ? 'checked' : '' ?> class="peer sr-only">
                                        <span class="inline-block px-3 py-1.5 text-xs font-bold border border-gray-200 rounded-lg bg-gray-50 peer-checked:bg-emerald-600 peer-checked:text-white peer-checked:border-emerald-600 transition-all select-none hover:bg-gray-100">
                                            <?= $c['name'] ?>
                                            <a href="plugins.php?tool=smart_cats_search&delete_cat=<?= $c['id'] ?>" onclick="return confirm('حذف الفئة؟')" class="mr-1 text-red-300 hover:text-white hover:bg-red-500 rounded px-1 transition text-[10px]">&times;</a>
                                        </span>
                                    </label>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        <button type="submit" name="update_cats" class="w-full bg-emerald-600 text-white py-2 rounded-lg text-sm font-bold hover:bg-emerald-700 transition shadow-md">حفظ الفئات</button>
                    </form>
                </div>
                <?php endforeach; ?>
            </div>
        </main>
    </body>
    </html>
    <?php
}

// ============================================================
// 4. حقن البيانات (Frontend Injection)
// ============================================================
function smart_cats_inject_meta($product) {
    $ids = json_decode($product['category_ids'] ?? '[]', true);
    // نضمن أن المصفوفة موجودة حتى لو فارغة لكي يراها الجافاسكريبت
    if (!is_array($ids)) $ids = [];
    
    // نقوم بطباعة العنصر دائماً حتى لو فارغ لضمان عمل كود البحث
    echo '<span class="smart-cat-meta hidden" style="display:none;" data-ids=\''.json_encode($ids).'\' data-title="'.htmlspecialchars($product['title']).'"></span>';
}
add_action('after_product_title', 'smart_cats_inject_meta');

// ============================================================
// 5. واجهة البحث والفلتر (Frontend Search & Filter)
// ============================================================
function smart_cats_search_ui() {
    global $pdo;
    $cats = $pdo->query("SELECT * FROM smart_categories ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);
    ?>
    
    <div id="smart-filter-widget" class="w-full mb-6 p-1">
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-2 flex flex-col md:flex-row gap-3">
            
            <div class="relative flex-1">
                <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <svg class="w-5 h-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" /></svg>
                </div>
                <input type="text" id="smart-search-input" placeholder="بحث عن منتج..." 
                       class="w-full pr-10 pl-4 py-2.5 bg-gray-50 border border-gray-200 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:outline-none text-sm font-bold text-gray-700 transition">
            </div>

            <div class="w-full md:w-1/3">
                <div class="relative">
                    <select id="smart-cat-select" class="w-full appearance-none pl-8 pr-4 py-2.5 bg-gray-50 border border-gray-200 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:outline-none text-sm font-bold text-gray-700 cursor-pointer transition">
                        <option value="">كل الفئات</option>
                        <?php foreach($cats as $c): ?>
                            <option value="<?= $c['id'] ?>"><?= $c['name'] ?></option>
                        <?php endforeach; ?>
                    </select>
                    <div class="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none text-gray-500">
                        <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7" /></svg>
                    </div>
                </div>
            </div>

            <button onclick="resetSmartFilters()" class="text-xs md:text-sm text-red-500 font-bold px-4 hover:bg-red-50 rounded-lg transition">مسح</button>
        </div>
    </div>

    <script>
    document.addEventListener("DOMContentLoaded", function() {
        
        // 1. نقل البيانات من span المخفي إلى الكارد نفسه
        const metas = document.querySelectorAll('.smart-cat-meta');
        
        metas.forEach(meta => {
            let card = meta.closest('.product-card');
            if (card) {
                card.setAttribute('data-cat-ids', meta.getAttribute('data-ids'));
                card.setAttribute('data-search-title', meta.getAttribute('data-title').toLowerCase());
            }
        });

        // 2. منطق الفلترة (المصحح)
        const searchInput = document.getElementById('smart-search-input');
        const catSelect = document.getElementById('smart-cat-select');

        function performFilter() {
            const term = searchInput.value.toLowerCase();
            const selectedCat = parseInt(catSelect.value) || 0;
            
            const cards = document.querySelectorAll('.product-card');
            
            cards.forEach(card => {
                // جلب مصفوفة الفئات
                let rawCats = JSON.parse(card.getAttribute('data-cat-ids') || "[]");
                
                // [التعديل الجوهري]: تحويل كل العناصر داخل المصفوفة إلى أرقام (Numbers) للمقارنة الصحيحة
                const catIds = rawCats.map(id => Number(id));
                
                const title = card.getAttribute('data-search-title') || "";

                // التحقق
                // هل المستخدم اختار "الكل" (0)؟ أو هل رقم الفئة موجود داخل مصفوفة المنتج؟
                const matchesCat = (selectedCat === 0) || (catIds.includes(selectedCat));
                
                // هل البحث فارغ؟ أو هل العنوان يحتوي على كلمة البحث؟
                const matchesText = (term === "") || (title.includes(term));

                if (matchesCat && matchesText) {
                    card.style.display = "flex"; 
                    card.classList.remove('opacity-50');
                } else {
                    card.style.display = "none";
                }
            });
        }

        if(searchInput) searchInput.addEventListener('input', performFilter);
        if(catSelect) catSelect.addEventListener('change', performFilter);

        window.resetSmartFilters = function() {
            if(searchInput) searchInput.value = "";
            if(catSelect) catSelect.value = "";
            performFilter();
        };
    });
    </script>
    <style>
        .custom-scroll::-webkit-scrollbar { width: 4px; }
        .custom-scroll::-webkit-scrollbar-thumb { background-color: #cbd5e1; border-radius: 4px; }
    </style>
    <?php
}
add_action('before_products', 'smart_cats_search_ui');
?>